#!/bin/bash -e
#
# Copyright 2023 RnD Center "ELVEES", JSC

set -eu

set_default() {
    MODE="0"
    INPUT="/tmp/01.264"
    PLANE_ID=31
    SENSOR="OV5647_0"
}

set_default

help() {
    cat << EOF
mcom03-demo-facesdk-pipeline - tool to run FaceSDK demo test
Usage: mcom03-demo-facesdk-pipeline [options]
Options:
    -h          Show help
    -i          Input file (default: $INPUT)
    -m          Mode:
                    0 - Playback from the file
                    1 - Playback from the sensor
                    (default: $MODE)
    -p          Parameter "plane-id" for kmssink.
                (set default for kernel 5.10: $PLANE_ID. For kernel 4.19 should be 28)
    -s          Sensor type:
                    OV2718, OV5647, IMX219, OV10823 sensors are supported
                    (default: $SENSOR)
EOF
}

while getopts "hi:m:p:s:" arg; do
    case $arg in
        h) set_default; help; exit 0;;
        i) INPUT=$OPTARG;;
        m) MODE=$OPTARG;;
        p) PLANE_ID=$OPTARG;;
        s) SENSOR=$OPTARG;;
        ?) exit 1;;
    esac
done

if [ "$MODE" = "0" ]; then
    # Checking the existence of the input file
    if [ ! -f "$INPUT" ]; then
        echo "Error: file $INPUT doesn't exist" >&2
        exit 1
    fi
elif [ "$MODE" = "1" ]; then
    # Adjust SETUPFILE and SENSORMODE
    if [[ "$SENSOR" = IMX219* ]]; then
        SETUPFILE="/etc/felix/imx219/imx219.cfg"
        SENSORMODE="2"
    elif [[ "$SENSOR" = OV2718* ]]; then
        SETUPFILE="/etc/felix/ov2718/ov2718.cfg"
        SENSORMODE="0"
    elif [[ "$SENSOR" = OV5647* ]]; then
        SETUPFILE="/etc/felix/ov5647/ov5647-arducam.cfg"
        SENSORMODE="0"
    elif [[ "$SENSOR" = OV10823* ]]; then
        SETUPFILE="/etc/felix/ov10823/ov10823.cfg"
        SENSORMODE="0"
    else
        echo "Error: unknown sensor name $SENSOR selected" >&2
        exit 1
    fi
else
    echo "Error: unknown mode ($MODE)" >&2
    exit 1
fi

if [ "$MODE" = "0" ]; then
        SRC="filesrc location=$INPUT ! h264parse ! omxh264dec"

elif [ "$MODE" = "1" ]; then
        SRC="felixsrc setup-file=$SETUPFILE sensor=$SENSOR sensor-mode=$SENSORMODE \
            exposure-auto=true awb-enable=true awb-algorithm=pid awb-mode=high-lum context=0 ! \
            queue max-size-buffers=1"
fi

gst-launch-1.0 --no-fault -v $SRC ! facesdk conf=/usr/share/facesdk/common-facesdk.conf \
    name=detector elcore-binary=/usr/share/facesdk/FaceSdkCapture detector.video_src ! \
    metadraw ! queue ! fpsdisplaysink text-overlay=false sync=false \
    video-sink="kmssink driver-name=mali-dp plane-id=$PLANE_ID"
