#!/usr/bin/env python3

# Copyright 2023 RnD Center "ELVEES", JSC

import argparse
import json
import pathlib
import re
import subprocess
import sys
import time
from typing import Tuple, Union

verbose = False


def _run(cmd, timeout: int = 1, disown: bool = False, check: bool = True) -> str:
    debug(f"\n# {cmd}")
    p = subprocess.Popen(
        cmd,
        shell=True,
        stdout=subprocess.PIPE,
        stderr=subprocess.PIPE,
    )

    if disown:
        return
    p.wait(timeout)
    stdout, _ = p.communicate()
    errcode = p.returncode
    if check and errcode:
        # Errorcode returned by command with timeout
        if errcode != 124:
            raise subprocess.CalledProcessError(errcode, cmd)
    return stdout.decode("utf-8").strip()


def debug(*foo: any) -> None:
    if verbose:
        print(" ".join(str(f) for f in foo))


def _get_sound_card_device_num(cardname: str) -> Tuple:
    output = _run(f'arecord -l | grep "{cardname}"')
    card = re.search(r"card \d+", output)
    card = card[0].strip("card ") if card else None
    device = re.search(r"device \d+", output)
    device = device[0].strip("device ") if device else None
    return card, device


def start_pipeline(
    alaw: Union[str, pathlib.Path],
    mulaw: Union[str, pathlib.Path],
    cardname: str,
    duration: int,
) -> None:
    card, device = _get_sound_card_device_num(cardname)
    _run(
        (
            f"timeout {duration} gst-launch-1.0 "
            f"alsasrc device=plughw:{card},{device} "
            f"! audioconvert "
            f"! audioresample "
            f"! audio/x-raw,rate=8000,channels=1 "
            "! tee name=law "
            "! queue "
            "! alawenc "
            "! wavenc "
            f"! filesink location={alaw} "
            "law. "
            "! queue "
            "! mulawenc "
            "! wavenc "
            f"! filesink location={mulaw} "
        ),
        timeout=duration + 5,
    )


def analyze(*args) -> int:
    def invoke_from_json(parameter):
        STREAM_NUM = 0
        if not ffprobe:
            return 0
        if "streams" not in ffprobe:
            return 0
        if parameter not in ffprobe["streams"][STREAM_NUM]:
            return 0
        return ffprobe["streams"][STREAM_NUM][parameter]

    def get_codec_from_name(name):
        if "G711A" in str(name):
            return "pcm_alaw"
        elif "G711U" in str(name):
            return "pcm_mulaw"
        else:
            return None

    # Source: https://en.wikipedia.org/wiki/G.711
    g711 = {
        "bit_rate": 64000,  # bit/s
        "bits_per_sample": 8,
        "channels": 1,
        "sample_rate": 8000,  # Hz
    }

    error = 0
    for file in args:
        ffprobe = json.loads(
            _run(f"ffprobe -v quiet -hide_banner -print_format json -show_streams {file}")
        )
        g711["codec_name"] = get_codec_from_name(file)
        for p in ["codec_name", "sample_rate", "channels", "bits_per_sample", "bit_rate"]:
            debug(f"{file}: expected {p} = {g711[p]}; obtained {p} = {invoke_from_json(p)}")
            if str(g711[p]) == str(invoke_from_json(p)):
                continue
            print(f"Wrong parameter {p} = {invoke_from_json(p)} in {file}")
            error += 1
    if not error:
        print("All checks completed successfully")
    return error


__doc__ = """Utility records files G.711A (alaw, a-law) and G.711U (mulaw, mu-law) and compares
video properies with reference.
"""

__usage__ = """
law-recorder.py -c "mfbsp-i2s" -o records - record files via sound card mfbsp-i2s and place them
  into records/.
"""


class Formatter(argparse.RawTextHelpFormatter, argparse.ArgumentDefaultsHelpFormatter):
    pass


def main():
    parser = argparse.ArgumentParser(
        description=__doc__, usage=__usage__, formatter_class=Formatter
    )
    parser.add_argument("-c", "--audio-card", required=True, help="audio card number")
    parser.add_argument("-f", "--filename", default="rec", help="filename without extension")
    parser.add_argument(
        "-o",
        "--output-dir",
        type=pathlib.Path,
        default=pathlib.Path().cwd(),
        help="directory for placing records",
    )
    parser.add_argument(
        "-v",
        "--verbose",
        default=False,
        action="store_true",
        help="increase verbosity, e.g., show called commands",
    )
    parser.add_argument("-d", "--duration", default=10, type=int, help="record duration")
    args = parser.parse_args()

    global verbose
    if args.verbose:
        verbose = True

    if not args.output_dir.exists():
        args.output_dir.mkdir(parents=True)

    alaw_record_file = args.output_dir / f"{args.filename}.record.G711A.wav"
    mulaw_record_file = args.output_dir / f"{args.filename}.record.G711U.wav"

    start_pipeline(
        alaw=alaw_record_file,
        mulaw=mulaw_record_file,
        cardname=args.audio_card,
        duration=args.duration,
    )
    debug(
        "Recordered files:",
        alaw_record_file,
        mulaw_record_file,
    )
    time.sleep(1)
    return analyze(mulaw_record_file, mulaw_record_file)


if __name__ == "__main__":
    sys.exit(main())
