# Copyright 2019-2022 RnD Center "ELVEES", JSC
# SPDX-License-Identifier: BSD-3-Clause
#
#.rst:
# ElcoreProject
# -------------
#
# Helper function for building Elcore targets as external projects.
#
# ``ELCORE_CMAKE_TOOLCHAIN_FILE`` should point to CMake toolchain file for Elcore.
# ``ELCORE_EXTRA_CMAKE_ARGS`` may point to CMake arguments for Elcore passed together with
# ``CMAKE_ARGS``.
#
#
# This module defines the following function:
#
# .. command:: elcoreproject_add
#
#   The ``elcoreproject_add()`` creates ExternalProject that will be built with Elcore toolchain.
#
#   elcoreproject_add(<name> <source_dir> [<option>...])
#
#   **Required Arguments:**
#     ``<name>``
#       Name of external project to add. ``<name>`` can be used functions from ExternalProject
#       module like ``ExternalProject_Get_Property`` or as dependency for targets.
#
#     ``<source_dir>``
#       Path to directory containing sources of the project to build. The path may be absolute or
#       relative to ``CMAKE_CURRENT_SOURCE_DIR``.
#
#   **Optional Arguments:**
#     ``CMAKE_ARGS <args>``
#       Arguments passed to the subproject.
#
#   ``elcoreproject_add()`` sets variable ``<name>_INSTALL_DIR`` to installation path of
#   subproject.

include(ExternalProject)
function(elcoreproject_add name source_dir)
    if(NOT EXISTS ${ELCORE_CMAKE_TOOLCHAIN_FILE})
        message(SEND_ERROR "CMake toolchain file not found. Following path was specified: \
                '${ELCORE_CMAKE_TOOLCHAIN_FILE}'")
    endif()

    if(NOT IS_ABSOLUTE ${source_dir})
        set(source_dir ${CMAKE_CURRENT_SOURCE_DIR}/${source_dir})
    endif()
    cmake_parse_arguments(ELCORE_PROJECT "" "" "CMAKE_ARGS" ${ARGN})

    # Extract CMAKE_<LANG>_FLAGS from ELCORE_PROJECT_CMAKE_ARGS
    foreach(ELCORE_ARG IN LISTS ELCORE_PROJECT_CMAKE_ARGS)
        if(ELCORE_ARG MATCHES "^-DCMAKE_C_FLAGS=.*$")
            string(REPLACE "-DCMAKE_C_FLAGS=" "" DSP_CMAKE_C_FLAGS ${ELCORE_ARG})
            list(REMOVE_ITEM ELCORE_PROJECT_CMAKE_ARGS ${ELCORE_ARG})
        endif()
        if(ELCORE_ARG MATCHES "^-DCMAKE_CXX_FLAGS=.*$")
            string(REPLACE "-DCMAKE_CXX_FLAGS=" "" DSP_CMAKE_CXX_FLAGS ${ELCORE_ARG})
            list(REMOVE_ITEM ELCORE_PROJECT_CMAKE_ARGS ${ELCORE_ARG})
        endif()
        if(ELCORE_ARG MATCHES "^-DCMAKE_ASM_FLAGS=.*$")
            string(REPLACE "-DCMAKE_ASM_FLAGS=" "" DSP_CMAKE_ASM_FLAGS ${ELCORE_ARG})
            list(REMOVE_ITEM ELCORE_PROJECT_CMAKE_ARGS ${ELCORE_ARG})
        endif()
    endforeach()

    # Merge CMake options from ELCORE_EXTRA_CMAKE_ARGS.
    # Options from ELCORE_EXTRA_CMAKE_ARGS have higher priority compared to options
    # of the same name in ELCORE_PROJECT_CMAKE_ARGS.
    if(ELCORE_EXTRA_CMAKE_ARGS)
        string(REPLACE "-D" ";-D" ELCORE_EXTRA_CMAKE_ARGS_LIST ${ELCORE_EXTRA_CMAKE_ARGS})
        foreach(ELCORE_ARG IN LISTS ELCORE_EXTRA_CMAKE_ARGS_LIST)
            if(ELCORE_ARG MATCHES "^-DCMAKE_C_FLAGS=.*$")
                string(REPLACE "-DCMAKE_C_FLAGS=" "" TMP_C_FLAGS ${ELCORE_ARG})
                set(DSP_CMAKE_C_FLAGS "${DSP_CMAKE_C_FLAGS} ${TMP_C_FLAGS}")
            elseif(ELCORE_ARG MATCHES "^-DCMAKE_CXX_FLAGS=.*$")
                string(REPLACE "-DCMAKE_CXX_FLAGS=" "" TMP_CXX_FLAGS ${ELCORE_ARG})
                set(DSP_CMAKE_CXX_FLAGS "${DSP_CMAKE_CXX_FLAGS} ${TMP_CXX_FLAGS}")
            elseif(ELCORE_ARG MATCHES "^-DCMAKE_ASM_FLAGS=.*$")
                string(REPLACE "-DCMAKE_ASM_FLAGS=" "" TMP_ASM_FLAGS ${ELCORE_ARG})
                set(DSP_CMAKE_ASM_FLAGS "${DSP_CMAKE_ASM_FLAGS} ${TMP_ASM_FLAGS}")
            else()
                list(POP_BACK ELCORE_PROJECT_CMAKE_ARGS ${ELCORE_ARG})
            endif()
        endforeach()
    endif()

    # Now ELCORE_PROJECT_CMAKE_ARGS contains only options differ from CMAKE_<LANG>_FLAGS

    # Use -Wall by default
    set(DSP_CMAKE_C_FLAGS "-Wall -Wno-unused-command-line-argument ${DSP_CMAKE_C_FLAGS}")
    set(DSP_CMAKE_CXX_FLAGS "-Wall -Wno-unused-command-line-argument ${DSP_CMAKE_CXX_FLAGS}")
    set(DSP_CMAKE_ASM_FLAGS "-Wall -Wno-unused-command-line-argument ${DSP_CMAKE_ASM_FLAGS}")

    externalproject_add(${name}
                        SOURCE_DIR ${source_dir}
                        CMAKE_ARGS -DCMAKE_TOOLCHAIN_FILE=${ELCORE_CMAKE_TOOLCHAIN_FILE}
                                    -DCMAKE_BUILD_TYPE=${CMAKE_BUILD_TYPE}
                                    -DCMAKE_INSTALL_PREFIX=<INSTALL_DIR>
                                    -DCMAKE_C_FLAGS=${DSP_CMAKE_C_FLAGS}
                                    -DCMAKE_CXX_FLAGS=${DSP_CMAKE_CXX_FLAGS}
                                    -DCMAKE_ASM_FLAGS=${DSP_CMAKE_ASM_FLAGS}
                                    ${ELCORE_PROJECT_CMAKE_ARGS}
                        TEST_EXCLUDE_FROM_MAIN TRUE
                        BUILD_ALWAYS TRUE)
    externalproject_get_property(${name} INSTALL_DIR)
    set(${name}_INSTALL_DIR ${INSTALL_DIR} PARENT_SCOPE)
endfunction()
